#ifndef __CWindow__
#define __CWindow__

#include <Testing/CTrace.hpp>
#include <Host/CDLLLoader.hpp>
#include <Collections/TPointerCollection.hpp>
#include "../Basics/GUIDefines.h"
#include "../Basics/ITimedObject.hpp"
#include "../Controls/CControlRoot.hpp"
#include "IWindow.hpp"
#include "IToolTipSupportingWindow.hpp"
#include "IToolTip.hpp"
#include "CWindowTools.hpp"

//	===========================================================================

using Exponent::Host::CDLLLoader;
using Exponent::GUI::Basics::ITimedObject;
using Exponent::GUI::Windowing::IWindow;
using Exponent::GUI::Windowing::IToolTipSupportingWindow;
using Exponent::GUI::Windowing::IToolTip;
using Exponent::GUI::Windowing::CWindowTools;
using Exponent::GUI::Controls::CControlRoot;
using Exponent::Testing::CTrace;
using Exponent::Collections::TPointerCollection;

//	===========================================================================

/**
 * @cond
 */
#ifdef WIN32
	#pragma comment (lib, "Comctl32.lib")
#endif
/**
 * @endcond
 */

namespace Exponent
{
	namespace GUI
	{
		namespace Windowing
		{
			/**
			 * @class CWindow CWindow.hpp
			 * @brief Implementation of a window.
			 *
			 * @date 20/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CWindow.hpp,v 1.11 2007/03/07 22:54:26 paul Exp $
			 */
			class CWindow : public IWindow, public IWindowChangeListener, public IFocusListener, public IToolTipSupportingWindow
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * Construction
				 * @param size The size of the window
				 */
				CWindow(const CRect &size);

				/**
				 * Destruction
				 */
				virtual ~CWindow();

//	===========================================================================

				/**
				 * Increment the reference count
				 */
				virtual void referenced();

				/**
				 * Decrement the reference count. Object is deleted if reference count is <= 0
				 */
				virtual void dereference();

				/**
				 * Get the reference count
				 * @retval long The reference count
				 */
				virtual long getReferenceCount() const;

				/**
				 * Get a description of the object
				 * @param string On return is filled with the description
				 * @param size The size of the stirng
				 */
				virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

				/**
				 * Set the parent window of this window
				 * @param parent The parent window
				 */
				virtual void setParentWindow(IWindow *parent) { m_windowParent = parent; }

				/**
				 * Get hte parent window
				 * @retval IWindow* The parent window, may be NULL
				 */
				virtual IWindow *getParentWindow() const { return m_windowParent; }

//	===========================================================================

				/**
				 * Register a child window
				 * @param child The child window
				 * @note For future use
				 */
				virtual void registerChildWindow(IWindow *child);

				/**
				 * Unregister all child windows
				 * @note For future use
				 */
				virtual void unregisterAllChildWindows();

				/**
				 * Unregister a specific child window
				 * @param child The child window to unregister
				 * @note For future use
				 */
				virtual void unregisterChldWindow(IWindow *child);

//	===========================================================================

				/**
				 * Get the widow handle
				 * @retval const SWindowHandle* The handle of this window
				 */
				virtual const SWindowHandle *getWindowHandle() const { return m_windowHandle; }

				/**
				 * Get the widow handle
				 * @retval SWindowHandle* The handle of this window
				 */
				virtual SWindowHandle *getMutableWindowHandle() const { return m_windowHandle; }

				/**
				 * Get the widow attributes
				 * @retval CWindowAttributes* The attributes of this window
				 */
				virtual CWindowAttributes *getWindowAttributes() const { return m_attributes; }

//	===========================================================================

				/**
				 * Initilalise the window
				 * @param windowAttributes The window attributes. (copy should be made, do not store this pointer - it may be deleted...)
				 * @param showWindow Should the window be shown immediately
				 */
				virtual void initialiseWindow(CWindowAttributes *windowAttributes, const bool showWindow);

				/**
				 * Uninitialise the window
				 */
				virtual void uninitialiseWindow();

//	===========================================================================

				/**
				 * Open the window
				 */
				virtual void openWindow();

				/**
				 * Close the widnow
				 */
				virtual void closeWindow();

				/**
				 * Update the window redraws entire window
				 */
				virtual void updateWindow();

				/**
				 * Disable the window
				 */
				virtual void disableWindow();

				/**
				 * Enable the window
				 */
				virtual void enableWindow();

				/**
				 * Window gained the focus
				 * @param window The window that lost focus
				 */
				virtual void gainedFocus(IWindow *window);

				/**
				 * Window lost the focus
				 * @param window The window that lost focus
				 */
				virtual void lostFocus(IWindow *window);

//	===========================================================================

				/**
				 * Draw the window images...
				 * @param graphics The graphics handle to use
				 */
				virtual void draw(CGraphics &graphics);

				/**
				 * Set an area of the window dirty
				 * @param area The area of the window relative to its top left to draw
				 */
				virtual void redrawWindow(const CRect &area);

				/**
				 * Set the whole window dirty - generally use sparingly!!
				 */
				virtual void redrawWindow();

				#ifndef WIN32
				/**
				 * Set the compositing control
				 * @param ref the user pane control
				 */
				void setCompositingControl(HIViewRef ref) { m_userWindow = ref; }
				#endif

//	===========================================================================

				/**
				 * Set the position of the window relative to its parent
				 * @param position The position relative to the parent, usually the desktop
				 */
				virtual void setWindowPosition(const CPoint &position);

				/**
				 * Get the postion relative to the parent
				 * @retval const CPoint& The window position relative to the parent, usually the desktop
				 */
				virtual const CPoint &getWindowPosition() const { return m_windowPosition; }

				/**
				 * Set the size of the widnwo
				 * @param dimension The dimension of the window
				 */
				virtual void setWindowSize(const CDimension &dimension);

				/**
				 * Get the size of the window
				 * @retval const CDimension& The dimension of the window
				 */
				virtual const CDimension &getWindowSize() const { return m_windowDimension; }

				/**
				 * Set the size and position
				 * @param positionAndSize The new position and size of the window
				 */
				virtual void setWindowPositionAndSize(const CRect &positionAndSize);

				/**
				 * Get the size and position
				 * @retval const CRect& The size and position of the window
				 * @see getWindowSize
				 * @see getWindowPosition
				 */
				virtual const CRect &getWindowPositionAndSize() const { return m_windowSize; }

				/**
				 * Set the window alpha value
				 * @param alpha The alpha level 0 - 1
				 */
				virtual void setWindowAlpha(const double alpha);

//	===========================================================================

				/**
				 * Handle a timer event happening - Intended for internal use
				 * @param id The id of the timer that has fired an event
				 */
				virtual void handleTimerEvent(const long id);

				/**
				 * Start a timer
				 * @param id The unique id of the timer object to start
				 * @param timeInMilliseconds The time in milliseconds between updates
				 */
				virtual void startTimer(const long id, const long timeInMilliseconds);

				/**
				 * Stop a timer
				 * @param id The unique id of the timer
				 */
				virtual void stopTimer(const long id);

				/**
				 * Add a new timed object
				 * @param timedObject The object to add to the timer collection
				 */
				virtual void addTimedObject(ITimedObject *timedObject);

				/**
				 * Remove a timed objects
				 * @param timedObject The object to remove from timing
				 */
				virtual void removeTimedObject(ITimedObject *timedObject);

				/**
				 * Clear the timed bojects
				 */
				virtual void clearTimedObjects();

//	===========================================================================

				/**
				 * Register a focus listener
				 * @param listener The listener for window focus information
				 */
				virtual void registerFocusListener(IFocusListener *listener) { m_focusListener = listener; }

				/**
				 * Register a window change listener
				 * @param listener The listener for window change information
				 */
				virtual void registerWindowChangeListener(IWindowChangeListener *listener) { m_windowChangeListener = listener; }

				/**
				 * Register a mouse listener
				 * @param listener The listener for mouse information
				 */
				virtual void registerMouseListener(IMouseListener *listener) { m_mouseListener = listener; }

				/**
				 * Register a drop file listener
				 * @param listener The listener for drop file information
				 */
				virtual void registerDropFileListener(IDropFileListener *listener) { m_dropFileListener = listener; }

				/**
				 * Register keyboard listener
				 * @param listener The listener for keyboard information
				 */
				virtual void registerKeyboardListener(IKeyboardListener *listener) { m_keyboardListener = listener; }

//	===========================================================================

				/**
				 * Can the window handle dropped files??
				 * @retval bool True if window can handle mouse events, false otherwise
				 */
				virtual bool isMouseEnabled() const { return m_mouseListener != NULL; }

				/**
				 * Can the window handle dropped files??
				 * @retval bool True if window can handle keyboard events, false otherwise
				 */
				virtual bool isKeyboardEnabled() const { return m_keyboardListener != NULL; }

				/**
				 * Can the window handle dropped files??
				 * @retval bool True if window can handle dropped files, false otherwise
				 */
				virtual bool isDropFileEnabled() const { return m_dropFileListener != NULL; }

//	===========================================================================

				// Set the tooltip to display
				virtual void setToolTip(IToolTip *toolTip = NULL) { m_toolTip = toolTip; }

				// Get the mutable tool tip
				virtual IToolTip *getMutableToolTip() const { return m_toolTip; }

				// Get the tool tip
				virtual const IToolTip *getToolTip() const { return m_toolTip; }

//	===========================================================================

				/**
				 * Get the mouse
				 * @retval CMouse* The mouse for this window
				 */
				virtual CMouse *getMouse() { return &m_mouse; }

				/**
				 * Get the control root
				 * @retval IControlRoot* The control root of this window
				 */
				virtual IControlRoot *getControlRoot() const { return m_theControlRoot; }

//	===========================================================================

				/**
				 * Window has been resized
				 * @param window The window that was changed
				 * @param newDimension The new size of the window
				 */
				virtual void windowSized(IWindow *window, const CDimension &newDimension);

				/**
				 * Window has been moved
				 * @param window The window that was changed
				 * @param newTopLeft The new top left of the window
				 */
				virtual void windowMoved(IWindow *window, const CPoint &newTopLeft);

				/**
				 * Window has been closed
				 * @param window The window that was changed
				 * @retval bool Return true to close, return false to cancel closure
				 */
				virtual bool windowClosed(IWindow *window);

				/**
				 * Window has been opened
				 * @param window The window that was changed
				 */
				virtual void windowOpened(IWindow *window);

				/**
				 * Window has been minimised
				 * @param window The window that was changed
				 */
				virtual void windowMinimised(IWindow *window);

				/**
				 * Window has been maximised
				 * @param window The window that was changed
				 */
				virtual void windowMaximised(IWindow *window);

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Create the window
				 */
				virtual bool createWindow();

				/**
				 * Destroy the widnow
				 */
				virtual void destroyWindow();

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					/**
					 * Handle window events
					 * @param windowHandle The handle to the window
					 * @param message The message to handle
					 * @param wParam The high bits
					 * @param lParam The low bits
					 * @retval LRESULT the result, did we handle?
					 */
					virtual LRESULT handleWindowEvents(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam);

					/**
					 * Handle key hook events
					 * @param nCode The key code
					 * @param wParam The high bits
					 * @param lParam The low bits
					 * @retval LRESULT the result, did we handle?
					 * @see Windows hooks
					 */
					virtual LRESULT handleKeyboardEvents(int nCode, WPARAM wParam, LPARAM lParam);

					/**
					 * Handle key hook events
					 * @param nCode The key code
					 * @param wParam The high bits
					 * @param lParam The low bits
					 * @retval LRESULT the result, did we handle?
					 * @see Windows hooks
					 */
					LRESULT handleMouseWheelEvents(int nCode, WPARAM wParam, LPARAM lParam);

					/**
					 * Construct a mouse position
					 * @param lParam The windows mouse position
					 */
					virtual void constructMousePosition(LPARAM lParam);

					/**
					 * Start a track mouse event
					 */
					virtual void startTrackMouseEvent();

					/**
					 * Unhook the keyboard handler
					 */
					virtual void unhookKeyboardHandler();

					/**
					 * Unhook the wheel handler
					 */
					void unhookWheelHandler();
				/** @cond */
				#else
				/** @endcond */
					/**
					 * Handle window events
					 * @param handler The window handler
					 * @param theEvent The mac window event
					 * @retval pascal OSErr The error status, did we handle?
					 */
					virtual pascal OSStatus handleWindowEvents(EventHandlerCallRef handler, EventRef theEvent);

					/**
					 * Handle a drag and drop event
					 * @param window The window dropped to
					 * @param dragReference A reference to the dragged object
					 * @retval pascal OSErr The error status, did we handle?
					 */
					virtual pascal OSErr handleDropEvent(WindowRef window, DragRef dragReference);

					/**
					 * Construct the mouse position
					 * @param theEvent The mac mouse event
					 */
					virtual void constructMousePosition(EventRef theEvent);

					/**
					 * Construct the keyboard event
					 * @param theEvent The mac key board event
					 */
					virtual void constructKeyboardEvent(EventRef theEvent);
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					/**
					 * Main proc call back function for windows
					 * @param windowHandle The handle to the window
					 * @param message The message to handle
					 * @param wParam The high bits
					 * @param lParam The low bits
					 * @retval LRESULT the result, did we handle?
					 */
					static LRESULT CALLBACK WindowProc(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam);

					/**
					 * Handle key hook events
					 * @param nCode The key code
					 * @param wParam The high bits
					 * @param lParam The low bits
					 * @retval LRESULT the result, did we handle?
					 * @see Windows hooks
					 */
					static LRESULT CALLBACK KeyboardProc(int nCode, WPARAM wParam, LPARAM lParam);

					/**
					 * Handle wheel hook events
					 * @param nCode The key code
					 * @param wParam The high bits
					 * @param lParam The low bits
					 * @retval LRESULT the result, did we handle?
					 * @see Windows hooks
					 */
					static LRESULT CALLBACK MouseWheelProc(int nCode, WPARAM wParam, LPARAM lParam);

					/**
					 * Handle timer events
					 * @param windowHandle The handle to the window being timed
					 * @param message The message
					 * @param eventID The id of the timer event
					 * @param dwTime The time position
					 * @see TimerProc in platform SDK
					 */
					static VOID CALLBACK TimerProc(HWND windowHandle, UINT message, UINT_PTR eventID, DWORD dwTime);
				/** @cond */
				#else
				/** @endcond */
					/**
					 * Main proc call back function for window
					 * @param handler The event handler
					 * @param theEvent The event to handler
					 * @param userData The window user data
					 * @retval pascal OSStatus The error status, did we handle?
					 * @note intended for internal use
					 */
					static pascal OSStatus WindowProc(EventHandlerCallRef handler, EventRef theEvent, void *userData);

					/**
					 * Handle drag and drop messages
					 * @param window The window that had a drop message
					 * @param userData The user data assocuated with the window
					 * @param dragReference The object dropped
					 * @retval pascal OSErr The error status, did we handle?
					 * @note intended for internal use
					 */
					static pascal OSErr handleDropMessages(WindowRef window, void *userData, DragRef dragReference);

					/**
					 * Handle timer events
					 * @param timer The timer that expired
					 * @param info The timer info
					 * @note intended for internal use
					 */
					static void handleTimerMessages(CFRunLoopTimerRef timer, void *info);
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

				/** @cond */
				#ifndef WIN32
				/** @endcond */

					/**
					 * @class CTimerObject CWindow.hpp
					 * @brief Mac Specific inner timer object, ties together the CFTimer stuff and our timers
					 */
					class CTimerObject : public CCountedObject
					{
					public:

						/**
						 * Construction
						 */
						CTimerObject()
						{
							m_isActive = false;
							NULL_POINTER(m_timer);
						}

						/**
						 * Destruction
						 */
						virtual ~CTimerObject()
						{
							if (m_timer)
							{
								CFRelease(m_timer);
							}
						}

						CFRunLoopTimerRef m_timer;							/**< Timer context */
						bool m_isActive;									/**< True if the timer is running, false otherwise */
					};
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

				IMouseListener *m_mouseListener;							/**< Mouse listener */
				IDropFileListener *m_dropFileListener;						/**< Drop file listener */
				IKeyboardListener *m_keyboardListener;						/**< Keyboatd listener */
				IWindowChangeListener *m_windowChangeListener;				/**< Window changed listener */
				IFocusListener *m_focusListener;							/**< Window focus changed listener */

				CWindowAttributes *m_attributes;							/**< Attributes of the window */
				SWindowHandle *m_windowHandle;								/**< Handle of the window */
				IWindow *m_windowParent;									/**< The parent of this window */

				CDimension m_windowDimension;								/**< The dimensions of the window */
				CPoint m_windowPosition;									/**< The position (globally) of the window */
				CRect m_windowSize;											/**< The widow size */

				CMouse m_mouse;												/**< The mouse */
				CPoint m_mousePoint;										/**< Current mouse position */
				CGraphics m_graphics;										/**< The graphics object */

				CMouseEvent m_mouseEvent;									/**< The mouse event */
				CKeyboardEvent m_keyEvent;									/**< The keyboard event */

				bool m_windowInitialised;									/**< Is the window intialised? */

				IToolTip *m_toolTip;										/**< The tool tip for this window */

				TPointerCollection<ITimedObject> *m_timedObjects;			/**< The timed objects */

				CControlRoot *m_theControlRoot;								/**< The root control that all window have */

				long m_referenceCount;										/**< How many things point to this item? */

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					SETLAYERATTRIBUTE m_setLayeredWindowAttributes;			/**< The layered window function */
					HHOOK m_keyboardHook;									/**< The hook for keyboard events */
					HHOOK m_mouseHook;										/**< The hook for mouse wheel events */
					static HWND m_hookWindowHandle;							/**< The window hook */
				/** @cond */
				#else
				/** @endcond */
					bool m_leftClick;										/**< Is this a left click? */
					TPointerCollection<CTimerObject> *m_timerContexts;		/**< The timer contexts.. */
					CRect m_redrawUpdateArea;								/**< The area to update */
					bool m_updateAreaIsValid;								/**< Do we draw with the update area? */
					HIViewRef m_userWindow;									/**< Hiview ref for the user pane */
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

			};
		}
	}
}
#endif	// End of CWindow.hpp